#include <napi.h>
#include <stdio.h>
#include <iostream>
#include "node_processfiledirectory.hpp"
#include "node_helpers.hpp"
#include "photochemistry/image.hpp"
#include "photochemistry/process.hpp"
#include "photochemistry/renderqueue.hpp"
#include "photochemistry/utf8.hpp"
#include "photochemistry/ioqueue.hpp"
#include <opencv2/opencv.hpp>
#include <future>
#include <iostream>
#include <memory>

using namespace photochemistry;



// Static initializaiton for bridging class tso JS
Napi::Object PCHProcessFileDirectory::Init(Napi::Env env, Napi::Object exports) {
    Napi::Function func =
        DefineClass(env, "ProcessFileDirectory", {
            InstanceMethod("reload", &PCHProcessFileDirectory::reload),
            InstanceMethod("hasChanged", &PCHProcessFileDirectory::hasChanged),
            InstanceMethod("listFiles", &PCHProcessFileDirectory::listFiles),
            InstanceMethod("getProcessFile", &PCHProcessFileDirectory::getProcessFile),
            InstanceMethod("getProcessFileProperties", &PCHProcessFileDirectory::getProcessFileProperties),
            InstanceMethod("setProcessFileProperties", &PCHProcessFileDirectory::setProcessFileProperties),
            InstanceMethod("applyProcessFileProperties", &PCHProcessFileDirectory::applyProcessFileProperties),
            InstanceMethod("adjustProcessFileAdjustment", &PCHProcessFileDirectory::adjustProcessFileAdjustment),
            InstanceMethod("getImageTexture16F", &PCHProcessFileDirectory::getImageTexture16F),
            InstanceMethod("exportImageFile", &PCHProcessFileDirectory::exportImageFile),
            InstanceMethod("flipHorizontally", &PCHProcessFileDirectory::flipHorizontally),
            InstanceMethod("rotateClockwise", &PCHProcessFileDirectory::rotateClockwise),
            InstanceMethod("rotateCounterClockwise", &PCHProcessFileDirectory::rotateCounterClockwise),
            InstanceMethod("revertFiles", &PCHProcessFileDirectory::revertFiles),
            InstanceMethod("applyAutoSettingsWithoutSaving", &PCHProcessFileDirectory::applyAutoSettingsWithoutSaving),
            InstanceMethod("autoCropWithoutSaving", &PCHProcessFileDirectory::autoCropWithoutSaving),
            //InstanceMethod("getInputThumbnailImage", &PCHProcessFileDirectory::getInputThumbnailImage),
            //InstanceMethod("getImage", &PCHProcessFileDirectory::getImage),
        });
    Napi::FunctionReference* constructor = new Napi::FunctionReference();
    *constructor = Napi::Persistent(func);
    //env.SetInstanceData(constructor); TODO, allow multiple classes to set instance data
    exports.Set("ProcessFileDirectory", func);
    // sphotochemistry::ProcessAutoAdjuster::debugPath = "/Users/abe/code/filmlab/FilmLabDesktop/debug";
    return exports;
}

PCHProcessFileDirectory::PCHProcessFileDirectory(const Napi::CallbackInfo& info) :
  Napi::ObjectWrap<PCHProcessFileDirectory>(info) {
    Napi::Env env = info.Env();
    int length = info.Length();
    if (length <= 0 || !info[0].IsString()) {
        Napi::TypeError::New(env, "String argument for path expected").ThrowAsJavaScriptException();
        return;
    }

    std::string path = info[0].As<Napi::String>().Utf8Value();
    std::cout << "Initializing ProcessFileDirectory using path: " << path << std::endl;
    dir = photochemistry::storage::ProcessFileDirectory::open(path);
    dir->onChanged.connect([this](){
        this->_hasChanged = true;
    });
}

void PCHProcessFileDirectory::reload(const Napi::CallbackInfo& info) {
    dir->load();
}

Napi::Value PCHProcessFileDirectory::hasChanged(const Napi::CallbackInfo& info) {
    bool changed = _hasChanged;
    _hasChanged = false;
    return Napi::Boolean::New(info.Env(), changed);
}

Napi::Value PCHProcessFileDirectory::listFiles(const Napi::CallbackInfo& info) {
    Napi::Env env = info.Env();
    auto files = dir->getFiles();
    Napi::Array filesArray =
        Napi::Array::New(env, files.size());
    for (size_t i = 0; i < files.size(); i++) {
        Napi::Object fileInfo = Napi::Object::New(env);
        fileInfo.Set("filename", Napi::String::New(env, files[i].filename));
        fileInfo.Set("sourceFilename", Napi::String::New(env, files[i].sourceFilename));
        fileInfo.Set("modificationTime", Napi::Number::New(env, files[i].modificationTime));
        fileInfo.Set("createTime", Napi::Number::New(env, files[i].createTime));
        filesArray[i] = fileInfo;
    }
    return filesArray;
}

Napi::Value PCHProcessFileDirectory::getProcessFile(const Napi::CallbackInfo& info) {
    Napi::Env env = info.Env();
    if (info.Length() < 1 || !info[0].IsString()) {
        Napi::TypeError::New(env, "String argument for filename expected").ThrowAsJavaScriptException();
    }
    std::string filePath = info[0].As<Napi::String>().Utf8Value();
    std::cout << "Attempting to get process file at" << filePath << std::endl;
    auto pf = dir->getProcessFile(filePath);
    Napi::Object fileInfo = Napi::Object::New(env);
    fileInfo.Set("filename", Napi::String::New(env, pf->getFilename()));
    fileInfo.Set("sourceFilename", Napi::String::New(env, pf->getInputImageFilename()));
    fileInfo.Set("modificationTime", Napi::Number::New(env, pf->editTime.time_since_epoch().count()));
    fileInfo.Set("createTime", Napi::Number::New(env, pf->createTime.time_since_epoch().count()));
    return fileInfo;
}

Napi::Value PCHProcessFileDirectory::getProcessFileProperties(const Napi::CallbackInfo& info) {
    Napi::Env env = info.Env();
    if (info.Length() < 1 || !info[0].IsString()) {
        Napi::TypeError::New(env, "String argument for filename expected").ThrowAsJavaScriptException();
    }
    std::string path = info[0].As<Napi::String>().Utf8Value();
    printf("getProcessFileProperties using path: %s\n", path.c_str());
    auto pf = dir->getProcessFile(path);
    std::string json = pf->getProcessorProperties().dump();

    return Napi::String::New(env, json);
}

void PCHProcessFileDirectory::setProcessFileProperties(const Napi::CallbackInfo& info) {
    Napi::Env env = info.Env();
    if (info.Length() < 2 || !info[0].IsString() || !info[1].IsString()) {
        Napi::TypeError::New(env, "setProcessFileProperties, string arguments for filename and properties JSON expected").ThrowAsJavaScriptException();
    }
    auto p = dir->getProcessFile(info[0].As<Napi::String>().Utf8Value());
    std::string propertiesJSON = info[1].As<Napi::String>().Utf8Value();
    p->setProcessorProperties(nlohmann::json::parse(propertiesJSON));
    p->save();
}

void PCHProcessFileDirectory::applyProcessFileProperties(const Napi::CallbackInfo& info) {
    Napi::Array filenameInputArray = info[0].As<Napi::Array>();
    nlohmann::json propertiesJSON = nlohmann::json::parse(info[1].As<Napi::String>().Utf8Value());
    std::vector<std::string> filenames;
    for (uint32_t i = 0; i < filenameInputArray.Length(); i++) {
        filenames.push_back(filenameInputArray.Get(i).As<Napi::String>().Utf8Value());
    }

    Napi::Object autoChoices = info[2].As<Napi::Object>();
    std::cout << "autoChoices: " << autoChoices.ToString().Utf8Value() << std::endl;
    AutomaticOptions automaticOptions;
    if (autoChoices.Has("processAdjustments") && autoChoices.Get("processAdjustments").As<Napi::Boolean>().Value()) {
        automaticOptions.set(AutomaticOption::PROCESS_ADJUSTMENTS);
    }
    if (autoChoices.Has("crop") && autoChoices.Get("crop").As<Napi::Boolean>().Value()) {
        automaticOptions.set(AutomaticOption::CROP);
    }
    if (autoChoices.Has("orientation") && autoChoices.Get("orientation").As<Napi::Boolean>().Value()) {
        automaticOptions.set(AutomaticOption::ORIENTATION);
    }

    for (std::string filename: filenames) {
        photochemistry::IOQueue::parallelQueue()->run([=]{
            try {
                std::cout << "DEBUG: Applying process file properties to " << filename << " in queue" << std::endl;
                if (photochemistry::storage::ProcessFile::isProcessFile(filename)) {                
                    auto p = this->dir->getProcessFile(filename);
                    std::cout << "DEBUG: Updating process file " << filename << std::endl;
                    p->applyProcessorProperties(propertiesJSON, automaticOptions);
                    p->save();
                } else {
                    std::cout << "DEBUG: Making process file for source file " << filename << std::endl;
                    this->dir->makeProcessFileForSourceFile(filename, propertiesJSON, automaticOptions);
                }
                std::cout << "Completed: Process file properties applied to " << filename << " in queue" << std::endl;
            } catch (const std::exception& e) {
                std::cout << "Error applying process file properties to " << filename << ": " << e.what() << std::endl;
            }
       });
    }

}

void PCHProcessFileDirectory::adjustProcessFileAdjustment(const Napi::CallbackInfo& info) {
    Napi::Array filenameInputArray = info[0].As<Napi::Array>();
    photochemistry::ProcessAdjustmentFieldParadigm field = (photochemistry::ProcessAdjustmentFieldParadigm)(info[1].As<Napi::Number>().Int32Value());
    double steps = info[2].As<Napi::Number>().DoubleValue();
    std::vector<std::string> filenames;
    for (uint32_t i = 0; i < filenameInputArray.Length(); i++) {
        filenames.push_back(filenameInputArray.Get(i).As<Napi::String>().Utf8Value());
    }
    std::cout << "Hello world!\n";

    //std::thread([this, filenames, field, steps](){
        for (std::string filename: filenames) {
            std::cout << "Editing " << filename << "\n";
            auto p = this->dir->getProcessFile(filename);
            p->edit([&](std::shared_ptr<ImageProcessor> processor) {
                std::cout << "Making field adjustment for " << field << " by " << steps << "\n";
                processor->process->adjustField(field, steps);
            });
            p->save();
        }
    //}).detach();
}

void PCHProcessFileDirectory::flipHorizontally(const Napi::CallbackInfo& info) {
    auto p = dir->getProcessFile(info[0].As<Napi::String>().Utf8Value());
    p->flipHorizontally();
    p->save();
}

void PCHProcessFileDirectory::rotateClockwise(const Napi::CallbackInfo& info) {
    auto p = dir->getProcessFile(info[0].As<Napi::String>().Utf8Value());
    p->rotateClockwise();
    p->save();
}

void PCHProcessFileDirectory::rotateCounterClockwise(const Napi::CallbackInfo& info) {
    auto p = dir->getProcessFile(info[0].As<Napi::String>().Utf8Value());
    p->rotateCounterClockwise();
    p->save();
}

void PCHProcessFileDirectory::revertFiles(const Napi::CallbackInfo& info) {
    Napi::Array filenameInputArray = info[0].As<Napi::Array>();
    for (uint32_t i = 0; i < filenameInputArray.Length(); i++) {
        std::string filename = filenameInputArray.Get(i).As<Napi::String>().Utf8Value();
        dir->revert(filename);
    }
}

Napi::Value PCHProcessFileDirectory::applyAutoSettingsWithoutSaving(const Napi::CallbackInfo& info) {
    Napi::Env env = info.Env();
    auto pf = dir->getProcessFile(info[0].As<Napi::String>().Utf8Value());
    nlohmann::json propertiesJSON = nlohmann::json::parse(info[1].As<Napi::String>().Utf8Value());
    pf->applyProcessorProperties(propertiesJSON, AutomaticOptions::processAdjustmentsOnly());
    auto newProperties = pf->getProcessorProperties();
    return Napi::String::New(env, newProperties.dump());
}

Napi::Value PCHProcessFileDirectory::autoCropWithoutSaving(const Napi::CallbackInfo& info) {
    Napi::Env env = info.Env();
    auto pf = dir->getProcessFile(info[0].As<Napi::String>().Utf8Value());
    nlohmann::json propertiesJSON = nlohmann::json::parse(info[1].As<Napi::String>().Utf8Value());
    AutomaticOptions automaticOptions;
    automaticOptions.set(AutomaticOption::CROP);
    pf->applyProcessorProperties(propertiesJSON, automaticOptions);
    auto newProperties = pf->getProcessorProperties();
    return Napi::String::New(env, newProperties.dump());
}

// Need to be redone with new TypedThreadSafeFunction
// void PCHProcessFileDirectory::getImage(const Napi::CallbackInfo& info) {
//     Napi::Env env = info.Env();
//     auto pf = dir->getProcessFile(info[0].As<Napi::String>().Utf8Value());
//     Napi::Function jsCallback = info[1].As<Napi::Function>();
//     auto threadSafeCallback = Napi::ThreadSafeFunction::New(env, jsCallback, "getImageCallback", 0, 1);

//     pf->getInputThumbnail([=](photochemistry::Image image) mutable {
//       auto cppCallback = [image]( Napi::Env env, Napi::Function jsCallback, photochemistry::Image* unusedImagePtr) {
//         jsCallback.Call({ImageWrapper(env, image)});
//       };
//       threadSafeCallback.Acquire();
//       threadSafeCallback.BlockingCall(&image, cppCallback);
//       threadSafeCallback.Release();
//     });
// }

void convertTo16BitFloatingPoint(cv::Mat &mat) {
    switch(mat.depth()) {
    case CV_16U:
        mat.convertTo(mat, CV_16F, 1.0/65565.0);
        break;
    case CV_16F:
        break;
    case CV_8U:
        mat.convertTo(mat, CV_16F, 1.0/255.0);
        break;
    case CV_32F:
        mat.convertTo(mat, CV_16F);
        break;
    default:
        throw std::domain_error("Unsupported cv::Mat type can't be made into texture.");
    }
}


void PCHProcessFileDirectory::exportImageFile(const Napi::CallbackInfo& info) {
    Napi::Env env = info.Env();
    std::string inputProcessFile = info[0].As<Napi::String>().Utf8Value();
    std::string outputDir = info[1].As<Napi::String>().Utf8Value();
    photochemistry::ImageFileFormat outputFormat = (photochemistry::ImageFileFormat)info[2].As<Napi::Number>().Int32Value();
    nlohmann::json optionsJSON;
    std::string optionsJSONString = info[3].As<Napi::String>().Utf8Value();
    if (optionsJSONString.length() > 0) {
        try {
            optionsJSON = nlohmann::json::parse(optionsJSONString);
            std::cout << "Parsed options JSON: " << optionsJSON.dump() << std::endl;
        } catch (const std::exception& e) {
            std::cout << "Error parsing options JSON: " << e.what() << std::endl;
        }
    }
    int outputColorSpace = info[4].As<Napi::Number>().Int32Value();
    Napi::Function jsCallback = info[5].As<Napi::Function>();

    Context* context = new Context(Napi::Persistent(info.This()));
    auto threadSafeFunction = Napi::TypedThreadSafeFunction<Context, double, callbackWithDouble>::New(
        env, jsCallback, "callback", 0, 1, context, defaultFinalizer);

    auto p = dir->getProcessFile(inputProcessFile);
    std::string inputFilePath = p->getInputImageFilename();
    std::string outputExtension = ".jpg";
    std::cout << "Output format check " << (int)outputFormat << std::endl;
    std::shared_ptr<ImageSaverOptions> options = nullptr;
    if (outputFormat == photochemistry::ImageFileFormat::TIFF) {
        outputExtension = ".tiff";
        auto tiffOptions = std::make_shared<TiffImageSaverOptions>();
        safeCopyJSONValue<int>(optionsJSON, "bitDepth", tiffOptions->bitDepth);
        options = tiffOptions;
    } else if (outputFormat == photochemistry::ImageFileFormat::JPEGXL) {
        auto jxlOptions = std::make_shared<JxlImageSaverOptions>();
        safeCopyJSONValue<bool>(optionsJSON, "lossless", jxlOptions->lossless);
        safeCopyJSONValue<int>(optionsJSON, "effort", jxlOptions->effort);
        safeCopyJSONValue<double>(optionsJSON, "distance", jxlOptions->distance);
        options = jxlOptions;
        outputExtension = ".jxl";
    } else {
        // Jpeg
        auto jpegOptions = std::make_shared<JpegImageSaverOptions>();
        safeCopyJSONValue<int>(optionsJSON, "quality", jpegOptions->quality);
        options = jpegOptions;
    }
    const fs::path outputPath = utf8StringToPath(outputDir) / utf8StringToPath(inputFilePath + outputExtension);
    std::cout << "input path is " << inputFilePath << " and output path is " << outputPath << std::endl;
    
    const fs::path fullPath = dir->dir->fullPath(inputFilePath);
    const std::string fullPathString = pathToUTF8String(fullPath);
    Ref();

    photochemistry::IOQueue::serialQueue()->run([=]() {
        photochemistry::Image loadedImage;

        if (!photochemistry::ImageLoader::loadImageFromPath(fullPathString, loadedImage, false)) {
            std::cout << "Error, unable to load image from" << inputFilePath << std::endl;
            double* result = new double(0);
            // threadSafeFunction.Acquire();
            threadSafeFunction.BlockingCall(result);
            threadSafeFunction.Release();
        } else {
            p->editWithRenderer([&](std::shared_ptr<ImageProcessor> processor, std::shared_ptr<GLRenderer> renderer) {
                // Sometimes a newer raw engine may have improved color metadata
                // compared to the metadata we originally used for this image. But we
                // don't want the results to change unexpectedly after an upgrade.
                // So always use the stored color metadata
                loadedImage.format.colorSpaceRecords = processor->inputImageFormat.colorSpaceRecords;
                processor->setImage(loadedImage);
                std::cout << "Rendering to image file" << std::endl;
                processor->renderToImageFile(
                    renderer,
                    pathToUTF8String(outputPath),
                    (photochemistry::ColorSpace)outputColorSpace,
                    options);
                double* result = new double(1);
                // threadSafeFunction.Acquire();
                threadSafeFunction.BlockingCall(result);
                threadSafeFunction.Release();
            });
        }
        Unref();
    });

}

void PCHProcessFileDirectory::getImageTexture16F(const Napi::CallbackInfo& info) {

    std::locale("en_US.utf-8");

    Napi::Env env = info.Env();
    std::string filename = info[0].As<Napi::String>().Utf8Value();
    bool halfRes = info[1].As<Napi::Boolean>().Value();
    Napi::Function jsCallback = info[2].As<Napi::Function>();

    //std::cout << "Loading image texture " << filename << " from " << std::dynamic_pointer_cast<photochemistry::storage::FSDirectory>(dir->dir)->path << std::endl;

    Context* context = new Context(Napi::Persistent(info.This()));
    auto threadSafeFunction = Napi::TypedThreadSafeFunction<Context, photochemistry::Image, callbackWithImage>::New(
        env, jsCallback, "ImageTextureCallback", 0, 1, context, defaultFinalizer);

    std::string fullPath = pathToUTF8String(dir->dir->fullPath(filename));

    photochemistry::Image loadedImage;
    Ref(); // Hold a reference to the context so it doesn't get GC'd

    photochemistry::IOQueue::parallelQueue()->run([this, fullPath, halfRes, threadSafeFunction](){
        photochemistry::Image loadedImage;
        photochemistry::Image* heapImage = nullptr;
        if (photochemistry::ImageLoader::loadImageFromPath(fullPath, loadedImage, halfRes)) {
            heapImage = new photochemistry::Image(loadedImage);
            // Resize down to max 4096 on long side
            double sizeFactor = std::min(1.0,
                4096.0 / std::max(heapImage->mat.rows, heapImage->mat.cols));

            if (sizeFactor < 1.0) {
                if (heapImage->mat.depth() == CV_16F) {
                    // Unfortunate to have to use more memory, but CV_16F is not supported by cv::resize
                    heapImage->mat.convertTo(heapImage->mat, CV_32F);
                }
                std::cout << "Reducing large image from " << heapImage->mat.cols << "x" << heapImage->mat.rows <<
                    " to texture of size " << heapImage->mat.cols * sizeFactor << "x" << heapImage->mat.rows * sizeFactor << std::endl;
                cv::resize(heapImage->mat, heapImage->mat, cv::Size(), sizeFactor, sizeFactor, cv::INTER_AREA);
                heapImage->scale *= sizeFactor;
            }
            convertTo16BitFloatingPoint(heapImage->mat);
            std::cout << "Texture scale is " << heapImage->scale << std::endl;
        }
        //threadSafeFunction.Acquire();
        threadSafeFunction.NonBlockingCall(heapImage);
        threadSafeFunction.Release();
        Unref();
    });

}