const utils = require('@sentry/utils');
const electron = require('electron');
const electronNormalize = require('./electron-normalize.js');

let RENDERERS;
/**
 * Enable tracking of renderer properties via events
 *
 * This allows us to get the last known state of a renderer even if it's been destroyed
 **/
function trackRendererProperties() {
    if (RENDERERS) {
        return;
    }
    const renderers = (RENDERERS = new Map());
    function updateUrl(id, url) {
        const state = renderers.get(id) || { id };
        state.url = utils.normalizeUrlToBase(url, electron.app.getAppPath());
        renderers.set(id, state);
    }
    function updateTitle(id, title) {
        const state = renderers.get(id) || { id };
        state.title = title;
        renderers.set(id, state);
    }
    electronNormalize.onWebContentsCreated((contents) => {
        const id = contents.id;
        contents.on('did-navigate', (_, url) => updateUrl(id, url));
        contents.on('did-navigate-in-page', (_, url) => updateUrl(id, url));
        contents.on('page-title-updated', (_, title) => updateTitle(id, title));
        contents.on('destroyed', () => {
            // We need to delay since consumers of this API sometimes need to
            // access the state shortly after a renderer is destroyed
            setTimeout(() => {
                renderers.delete(id);
            }, 5000);
        });
    });
}
/** Gets the properties for a renderer */
function getRendererProperties(id) {
    return RENDERERS === null || RENDERERS === void 0 ? void 0 : RENDERERS.get(id);
}

exports.getRendererProperties = getRendererProperties;
exports.trackRendererProperties = trackRendererProperties;
//# sourceMappingURL=renderers.js.map
