const node = require('@sentry/node');
const utils = require('@sentry/utils');

/**
 * Normalizes all URLs in an event. See {@link normalizeUrl} for more
 * information. Mutates the passed in event.
 *
 * @param event The event to normalize.
 */
function normalizeEvent(event, basePath) {
    var _a, _b, _c, _d, _e;
    // Retrieve stack traces and normalize their paths. Without this, grouping
    // would not work due to usernames in file paths.
    for (const exception of ((_a = event.exception) === null || _a === void 0 ? void 0 : _a.values) || []) {
        for (const frame of ((_b = exception.stacktrace) === null || _b === void 0 ? void 0 : _b.frames) || []) {
            if (frame.filename) {
                frame.filename = utils.normalizeUrlToBase(frame.filename, basePath);
            }
        }
    }
    // We need to normalize debug ID images the same way as the stack frames for symbolicator to match them correctly
    for (const debugImage of ((_c = event.debug_meta) === null || _c === void 0 ? void 0 : _c.images) || []) {
        if (debugImage.type === 'sourcemap') {
            debugImage.code_file = utils.normalizeUrlToBase(debugImage.code_file, basePath);
        }
    }
    if (event.transaction) {
        event.transaction = utils.normalizeUrlToBase(event.transaction, basePath);
    }
    const { request = {} } = event;
    if (request.url) {
        request.url = utils.normalizeUrlToBase(request.url, basePath);
    }
    if (((_e = (_d = event.contexts) === null || _d === void 0 ? void 0 : _d.feedback) === null || _e === void 0 ? void 0 : _e.url) && typeof event.contexts.feedback.url === 'string') {
        event.contexts.feedback.url = utils.normalizeUrlToBase(event.contexts.feedback.url, basePath);
    }
    event.contexts = Object.assign(Object.assign({}, event.contexts), { runtime: {
            name: 'Electron',
            version: process.versions.electron,
        } });
    // The user agent is parsed by Sentry and would overwrite certain context
    // information, which we don't want. Generally remove it, since we know that
    // we are browsing with Chrome.
    if (request.headers) {
        delete request.headers['User-Agent'];
    }
    // The Node SDK includes server_name, which contains the machine name of the computer running Electron.
    // In this case this is likely to include PII.
    const { tags = {} } = event;
    delete tags.server_name;
    delete event.server_name;
    return event;
}
/** Normalizes URLs in any replay_event items found in an envelope */
function normalizeUrlsInReplayEnvelope(envelope, basePath) {
    let modifiedEnvelope = utils.createEnvelope(envelope[0]);
    let isReplay = false;
    utils.forEachEnvelopeItem(envelope, (item, type) => {
        var _a;
        if (type === 'replay_event') {
            isReplay = true;
            const [headers, event] = item;
            const currentScope = node.getCurrentScope().getScopeData();
            event.breadcrumbs = currentScope.breadcrumbs;
            event.tags = currentScope.tags;
            event.user = currentScope.user;
            if (Array.isArray(event.urls)) {
                event.urls = event.urls.map((url) => utils.normalizeUrlToBase(url, basePath));
            }
            if ((_a = event === null || event === void 0 ? void 0 : event.request) === null || _a === void 0 ? void 0 : _a.url) {
                event.request.url = utils.normalizeUrlToBase(event.request.url, basePath);
            }
            modifiedEnvelope = utils.addItemToEnvelope(modifiedEnvelope, [headers, event]);
        }
        else if (type === 'replay_recording') {
            modifiedEnvelope = utils.addItemToEnvelope(modifiedEnvelope, item);
        }
    });
    return isReplay ? modifiedEnvelope : envelope;
}
/**
 * Normalizes all URLs in a profile
 */
function normaliseProfile(profile, basePath) {
    for (const frame of profile.profile.frames) {
        if (frame.abs_path) {
            frame.abs_path = utils.normalizeUrlToBase(frame.abs_path, basePath);
        }
    }
}

exports.normaliseProfile = normaliseProfile;
exports.normalizeEvent = normalizeEvent;
exports.normalizeUrlsInReplayEnvelope = normalizeUrlsInReplayEnvelope;
//# sourceMappingURL=normalize.js.map
