import { __awaiter } from 'tslib';
import { defineIntegration, getCurrentScope, convertIntegrationFnToClass } from '@sentry/core';
import { dialog } from 'electron';

const INTEGRATION_NAME = 'OnUncaughtException';
/** Capture unhandled errors. */
const onUncaughtExceptionIntegration = defineIntegration(() => {
    return {
        name: INTEGRATION_NAME,
        setupOnce() {
            // noop
        },
        setup(client) {
            const options = client.getOptions();
            global.process.on('uncaughtException', (error) => {
                const scope = getCurrentScope();
                scope.addEventProcessor((event) => __awaiter(this, void 0, void 0, function* () {
                    return (Object.assign(Object.assign({}, event), { level: 'fatal' }));
                }));
                client.captureException(error, {
                    originalException: error,
                    data: {
                        mechanism: {
                            handled: false,
                            type: 'generic',
                        },
                    },
                }, scope);
                client.flush(options.shutdownTimeout || 2000).then(() => {
                    if (options === null || options === void 0 ? void 0 : options.onFatalError) {
                        options.onFatalError(error);
                    }
                    else if (global.process.listenerCount('uncaughtException') <= 2) {
                        // In addition to this handler there is always one in Electron
                        // The dialog is only shown if there are no other handlers
                        // eslint-disable-next-line no-console
                        console.error('Uncaught Exception:');
                        // eslint-disable-next-line no-console
                        console.error(error);
                        const ref = error.stack;
                        const stack = ref !== undefined ? ref : `${error.name}: ${error.message}`;
                        const message = `Uncaught Exception:\n${stack}`;
                        dialog.showErrorBox('A JavaScript error occurred in the main process', message);
                    }
                }, () => {
                    // ignore
                });
            });
        },
    };
});
/**
 * Capture unhandled errors.
 *
 * @deprecated Use `onUncaughtExceptionIntegration()` instead
 */
// eslint-disable-next-line deprecation/deprecation
const OnUncaughtException = convertIntegrationFnToClass(INTEGRATION_NAME, onUncaughtExceptionIntegration);

export { OnUncaughtException, onUncaughtExceptionIntegration };
//# sourceMappingURL=onuncaughtexception.js.map
