#include <napi.h>
#include "node_processfiledirectory.hpp"
#include "node_thumbnailcache.hpp"
#include "node_falloff.hpp"
#include "photochemistry/ioqueue.hpp"
#include "photochemistry/gl_headlesscontext.hpp"
#include "photochemistry/profiles/profileregistry.hpp"
#include "photochemistry/process.hpp"
using namespace photochemistry;

class PhotochemistryAddon : public Napi::Addon<PhotochemistryAddon> {
public:
    PhotochemistryAddon(Napi::Env env, Napi::Object exports) {
        // Initialize classes
        PCHProcessFileDirectory::Init(env, exports);
        PCHThumbnailCache::Init(env, exports);
        PCHFalloffProfile::Init(env, exports);

        // Define standalone functions
        DefineAddon(exports, {
            InstanceMethod("shutdownQueues", &PhotochemistryAddon::ShutdownQueues),
            InstanceMethod("getGLDevices", &PhotochemistryAddon::GetGLDevices),
            InstanceMethod("setGLDevice", &PhotochemistryAddon::SetGLDevice),
            InstanceMethod("getGLDevice", &PhotochemistryAddon::GetGLDevice),
            InstanceMethod("getProfiles", &PhotochemistryAddon::GetProfiles),
            InstanceMethod("setDefaultProcessProperties", &PhotochemistryAddon::SetDefaultProcessProperties),
        });
    }

private:
    Napi::Value GetGLDevices(const Napi::CallbackInfo& info) {
        return Napi::String::New(info.Env(), 
            nlohmann::json(photochemistry::GLHeadlessContextDeviceChooser::findAvailableDevices()).dump(2));
    }

    Napi::Value SetGLDevice(const Napi::CallbackInfo& info) {
        auto deviceInfo = nlohmann::json::parse(info[0].As<Napi::String>().Utf8Value());
        photochemistry::HeadlessRenderQueue::defaultQueue()->setGLDevice(deviceInfo);
        return Napi::Boolean::New(info.Env(), true);
    }

    Napi::Value GetGLDevice(const Napi::CallbackInfo& info) {
        return Napi::String::New(info.Env(), 
            nlohmann::json(photochemistry::HeadlessRenderQueue::defaultQueue()->getGLDevice()).dump());
    }

    void ShutdownQueues(const Napi::CallbackInfo& info) {
        photochemistry::IOQueue::deleteQueues();
        photochemistry::HeadlessRenderQueue::deleteDefaultQueue();
    }

    Napi::Value GetProfiles(const Napi::CallbackInfo& info) {
        return Napi::String::New(info.Env(), 
            nlohmann::json(photochemistry::profiles::SystemProfileRegistry::instance()->getProfilesJSON()).dump());
    }

    void SetDefaultProcessProperties(const Napi::CallbackInfo& info) {
        auto properties = nlohmann::json::parse(info[0].As<Napi::String>().Utf8Value());
        photochemistry::Process::defaultProperties = properties;
    }

  
        
};

NODE_API_ADDON(PhotochemistryAddon)
