import { __awaiter } from 'tslib';
import { getClient, captureEvent, createGetModuleFromFilename } from '@sentry/node';
import { watchdogTimer, stripSentryFramesAndReverse, callFrameToStackFrame, logger } from '@sentry/utils';
import { app } from 'electron';
import { Anr } from './integrations/anr.js';
import { sessionAnr } from './sessions.js';

function getRendererName(contents) {
    var _a, _b;
    const options = (_a = getClient()) === null || _a === void 0 ? void 0 : _a.getOptions();
    return (_b = options === null || options === void 0 ? void 0 : options.getRendererName) === null || _b === void 0 ? void 0 : _b.call(options, contents);
}
function sendRendererAnrEvent(contents, blockedMs, frames) {
    sessionAnr();
    const rendererName = getRendererName(contents) || 'renderer';
    const event = {
        level: 'error',
        exception: {
            values: [
                {
                    type: 'ApplicationNotResponding',
                    value: `Application Not Responding for at least ${blockedMs} ms`,
                    stacktrace: { frames },
                    mechanism: {
                        // This ensures the UI doesn't say 'Crashed in' for the stack trace
                        type: 'ANR',
                    },
                },
            ],
        },
        tags: {
            'event.process': rendererName,
        },
    };
    captureEvent(event);
}
function rendererDebugger(contents, pausedStack) {
    contents.debugger.attach('1.3');
    // Collect scriptId -> url map so we can look up the filenames later
    const scripts = new Map();
    const getModuleFromFilename = createGetModuleFromFilename(app.getAppPath());
    contents.debugger.on('message', (_, method, params) => {
        if (method === 'Debugger.scriptParsed') {
            const param = params;
            scripts.set(param.scriptId, param.url);
        }
        else if (method === 'Debugger.paused') {
            const param = params;
            if (param.reason !== 'other') {
                return;
            }
            // copy the frames
            const callFrames = [...param.callFrames];
            contents.debugger.sendCommand('Debugger.resume').then(null, () => {
                // ignore
            });
            const stackFrames = stripSentryFramesAndReverse(callFrames.map((frame) => callFrameToStackFrame(frame, scripts.get(frame.location.scriptId), getModuleFromFilename)));
            pausedStack(stackFrames);
        }
    });
    // In node, we enable just before pausing but for Chrome, the debugger must be enabled before he ANR event occurs
    contents.debugger.sendCommand('Debugger.enable').catch(() => {
        // ignore
    });
    return () => {
        return contents.debugger.sendCommand('Debugger.pause');
    };
}
let rendererWatchdogTimers;
function createHrTimer() {
    let lastPoll = process.hrtime();
    return {
        getTimeMs: () => {
            const [seconds, nanoSeconds] = process.hrtime(lastPoll);
            return Math.floor(seconds * 1e3 + nanoSeconds / 1e6);
        },
        reset: () => {
            lastPoll = process.hrtime();
        },
    };
}
/** Creates a renderer ANR status hook */
function createRendererAnrStatusHandler() {
    function log(message, ...args) {
        logger.log(`[Renderer ANR] ${message}`, ...args);
    }
    return (message, contents) => {
        rendererWatchdogTimers = rendererWatchdogTimers || new Map();
        let watchdog = rendererWatchdogTimers.get(contents);
        if (watchdog === undefined) {
            log('Renderer sent first status message', message.config);
            let pauseAndCapture;
            if (message.config.captureStackTrace) {
                log('Connecting to debugger');
                pauseAndCapture = rendererDebugger(contents, (frames) => {
                    log('Event captured with stack frames');
                    sendRendererAnrEvent(contents, message.config.anrThreshold, frames);
                });
            }
            watchdog = watchdogTimer(createHrTimer, 100, message.config.anrThreshold, () => __awaiter(this, void 0, void 0, function* () {
                log('Watchdog timeout');
                if (pauseAndCapture) {
                    log('Pausing debugger to capture stack trace');
                    pauseAndCapture();
                }
                else {
                    log('Capturing event');
                    sendRendererAnrEvent(contents, message.config.anrThreshold);
                }
            }));
            contents.once('destroyed', () => {
                rendererWatchdogTimers === null || rendererWatchdogTimers === void 0 ? void 0 : rendererWatchdogTimers.delete(contents);
            });
            rendererWatchdogTimers.set(contents, watchdog);
        }
        watchdog.poll();
        if (message.status !== 'alive') {
            log('Renderer visibility changed', message.status);
            watchdog.enabled(message.status === 'visible');
        }
    };
}
/**
 * @deprecated Use `Anr` integration instead.
 *
 * ```js
 * import { init, anrIntegration } from '@sentry/electron';
 *
 * init({
 *   dsn: "__DSN__",
 *   integrations: [anrIntegration({ captureStackTrace: true })],
 * });
 * ```
 */
function enableMainProcessAnrDetection(options = {}) {
    var _a;
    // eslint-disable-next-line deprecation/deprecation
    const integration = new Anr(options);
    const client = getClient();
    (_a = integration.setup) === null || _a === void 0 ? void 0 : _a.call(integration, client);
    return Promise.resolve();
}

export { createRendererAnrStatusHandler, enableMainProcessAnrDetection };
//# sourceMappingURL=anr.js.map
